"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Huawei Technologies Co., Ltd. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.activate = activate;
exports.deactivate = deactivate;
const vscode = require("vscode");
const lfDecoration = createDecorationType('LF', 'after');
const crDecoration = createDecorationType('CR', 'before');
let timeout;
function activate(context) {
    const triggerUpdateDecorations = () => {
        if (timeout) {
            clearTimeout(timeout);
        }
        if (show) {
            timeout = setTimeout(updateDecorations, 500);
        }
    };
    const updateDecorations = () => {
        if (!activeEditor || !show) {
            return;
        }
        const text = activeEditor.document.getText();
        activeEditor.setDecorations(lfDecoration, createDecorations(/\n/g, text, activeEditor));
        activeEditor.setDecorations(crDecoration, createDecorations(/\r/g, text, activeEditor));
    };
    let show = isActive();
    let activeEditor = vscode.window.activeTextEditor;
    if (activeEditor) {
        triggerUpdateDecorations();
    }
    vscode.window.onDidChangeActiveTextEditor(editor => {
        activeEditor = editor;
        if (editor && show) {
            triggerUpdateDecorations();
        }
    }, null, context.subscriptions);
    vscode.workspace.onDidChangeTextDocument(event => {
        if (!show || !activeEditor || event.document !== activeEditor.document) {
            return;
        }
        event.contentChanges.forEach(change => {
            if (change.text.indexOf('\n') < 0 && change.text.indexOf('\r') < 0) {
                return;
            }
            updateDecorations();
        });
        triggerUpdateDecorations();
    }, null, context.subscriptions);
    vscode.workspace.onDidChangeConfiguration(event => {
        if (!event.affectsConfiguration('editorLineEndings.showLineEndingsInEditor')) {
            return;
        }
        show = isActive();
        if (show) {
            updateDecorations();
        }
        else {
            clearDecorations();
        }
    }, null, context.subscriptions);
}
function deactivate() {
    clearDecorations();
}
function isActive() {
    return vscode.workspace.getConfiguration('editorLineEndings').get('showLineEndingsInEditor') || false;
}
function createDecorations(regEx, text, editor) {
    const ranges = [];
    for (let match = regEx.exec(text); match; match = regEx.exec(text)) {
        const position = editor.document.positionAt(match.index);
        ranges.push(new vscode.Range(position, position));
    }
    return ranges;
}
function clearDecorations() {
    if (timeout) {
        clearTimeout(timeout);
        timeout = undefined;
    }
    vscode.window.visibleTextEditors.forEach(editor => {
        editor.setDecorations(lfDecoration, []);
        editor.setDecorations(crDecoration, []);
    });
}
function createDecorationType(text, position) {
    const darkColor = 'rgba(255, 255, 255, 0.4)';
    const lightColor = 'rgba(0, 0, 0, 0.4)';
    return vscode.window.createTextEditorDecorationType({
        [position]: {
            border: '1px solid',
            contentText: text,
            margin: '0 0 0 0.3ch',
        },
        dark: {
            [position]: {
                borderColor: darkColor,
                color: darkColor,
            },
        },
        light: {
            [position]: {
                borderColor: lightColor,
                color: lightColor,
            },
        },
        rangeBehavior: vscode.DecorationRangeBehavior.ClosedClosed,
    });
}
//# sourceMappingURL=extension.js.map